# conf_inc.py
'''
各種設定値を保守するためのラッパースクリプト
  項目と値のセパレータ
    設定ファイルのコメント行を除く最初の行に
    sep <セパレータ>　で指定する。sepと<セパレータ>の間はスペースのみ
  一般の項目と値
    ＜項目＞ ＜セパレータ＞ ＜値＞　＜#コメント＞
    ※スペースを含む項目や値も許される
  複数行の値　値の位置からpythonのヒアドキュメント記述
    ＜項目＞ ＜セパレータ＞ ＜\'''値＞
    ＜値＞
    ...
    ＜値\'''＞
    ※ヒアドキュメント形式の値には行末コメントは不可
    ※また、上記の値にある\記号は表示上の都合で入れているだけ
  コメント
    #以降行末までの文字はコメント。
    ？ただし、pythonの例に習って \(エスケープ)#で#の文字とみなす？
  複数行のコメント
    ？スペースを除く行頭からpythonのヒアドキュメントで？
    そもそも定義行パターンに合致しないから無視される？

  値の更新
    単行項目の場合、行末にある既存のコメントは保持される
    値の更新はできてもコメントの更新には対応しない。
    コメントはエディターで直接編集すべし。
'''

from tkinter import *
from tkinter import messagebox as mbx
import re,os

separater = ':'  #セパレーター初期値
re_sep = 'sep\s+(.)\s*'   #セパレーターパターン
#re_item_all = '([^#]+?)\s+{}\s+(.*?)((\s*(#\w+))|)$' # {}は.format()で置換え
#項目名と値（値には行末のコメントも含む）
re_item_all = '([^#]+?)\s+{}\s+(.*?)$' # {}は.format()で置換え
#コメントを含む値から値とコメントを分離
re_comment = '(.*?)((\s*(#\w+))|)$'
re_item = '({})\s+{}\s+(.*?)$' # {}は.format()で置換え
#値のヒアドキュメント開始行判定
re_multi_st = "('''.*)$"
#値のヒアドキュメント終了行判定
re_multi_end = "(.*?''')$"

#設定ファイルの存在確認----------------------------
#設定ファイルがなければ作成
def existscheck_conf(fpath=None):
  try:
    if os.path.exists(fpath):
      return True
    else:
      #作成
      with open(fpath,mode='wt') as fp:
        #初期値を書き込み
        wt = 'sep {} #セパレーター定義 sep <separater_char>'.format(separater)
        fp.write(wt)
      return True
  except Exception as err:
    mbx.showerror('error',err)
    return False

#設定ファイルの全行を配列にして返す-----------------------
def get_lines(fpath=None):
  try:
    with open(fpath,mode='r') as fp:
      lines = fp.readlines()
  except Exception as err:
    mbx.showerror('error',err)
    return False
  return lines

#設定ファイルに定義されているセパレータを返す--------------
def get_separater(fpath):
  if existscheck_conf(fpath):
    lines = get_lines(fpath)
    if lines:
      for line in lines:
        res = re.search(re_sep,line.strip())
        if res:
          return res.group(1)
      return None


#設定ファイルを参照-------------------------------------------
#項目と値を辞書で返す
#fpath:設定ファイルパス
#item:抽出する項目名（同名項目は後出が優先）
#comm:True 値とセットで定義行の行末コメント、開始位置、終了位置も返す
#     False:値のみを返す
#戻り値：辞書形式
#  comm=False:{<項目名>:<値>,..}
#  comm=True :{<項目名>:[<値>,<コメント>,<index_st>,<index_end>],...}
def read_item_all(fpath=None,item=None,comm=False):
  if existscheck_conf(fpath):
    #セパレータの取得
    sep = get_separater(fpath)
    #reパターンにセパレータ挿入
    if item: #項目指定で抽出
      restr = re_item.format(item,sep)
    else:
      restr = re_item_all.format(sep)
    #全行配列に読込み
    lines = get_lines(fpath)
    
    confdic = {}
    multi = False  #ヒアドキュメント形式の値用のフラグ
    index_st = None  #定義行のリスト上の開始位置
    
    for idx,line in enumerate(lines):
      if multi:
        res = re.search(re_multi_end,line.strip())
        if res:
          #ヒアドキュメント記号は取り除く
          value += '\n'+res.group(1).replace("'''",'')
          if comm:  #コメント付き
            confdic[name] = [value,'',index_st,idx]
          else:
            confdic[name] = value
          multi = False
        else:
          value += '\n'+line
      else:
        res = re.search(restr,line.strip())
        if res:
          #セパレータ定義は無視
          if res.group(1) != 'sep':
            #値が複数行の場合が問題
            name = res.group(1)   #項目
            res_mu = re.search(re_multi_st,res.group(2))
            if not res_mu: #値がヒアドキュメントではない
              #コメント分離
              res_single = re.search(re_comment,res.group(2))
              value = res_single.group(1)  #値
              #辞書に登録
              if comm: #コメントも返す
                if res_single.group(4):
                  confdic[name] = [value,res_single.group(4),idx,idx]
                else:
                  confdic[name] = [value,'',idx,idx]
              else:
                confdic[name] = value
            else:  #ヒアドキュメント開始
              #ヒアドキュメント記号は取り除く
              value = res.group(2).replace("'''",'')
              index_st = idx  #ヒアドキュメント開始位置
              multi = True
    return confdic
  else:
    return None

#指定された項目の値を返す----------------------------------
#read_item_all()のラッパー
#item:項目名
#comm:True:値とコメントをリストで返す
#    :False:値だけを返す
#戻り値：辞書 {item:値|[値,コメント]}
def read_item(fpath=None,item=None,comm=False):
  return read_item_all(fpath,item,comm)


#指定された項目の値を変更する（追加も同じ）----------------
#戻り値：True | False
def update_item(fpath=None,item=None,data=None):
  #itemの登録状況を調べる
  itemdic = read_item(fpath,item,comm=True)
  sepstr = get_separater(fpath)
  #ヒアドキュメントデータなら記号で囲む
  if '\n' in data.strip():  #ヒアドキュメント
    data = "'''"+data+"'''"
  #置換えデータ作成
  wtxt = '\n{} {} {}'.format(item,sepstr,data)
  if not itemdic:
    #登録がないので新規項目として登録
    try:
      with open(fpath,mode='r') as fp:
        nowtxt = fp.read()
      with open(fpath,mode='w') as fp:
        fp.write(nowtxt.rstrip() + wtxt)
##      with open(fpath,mode='a') as fp:
##        fp.write(wtxt)
    except Exception as err:
      mbx.showerror('error',err)
      return False
  else:
    #通常定義行でコメントがある場合
    if itemdic[item][2] == itemdic[item][3] and itemdic[item][1]:
      wtxt += '  ' + itemdic[item][1]
    nowlist = get_lines(fpath)
    #定義行の前までのデータを結合
    wtxtbf = ''.join(nowlist[0:itemdic[item][2]])
    #定義行の後のデータを結合
    wtxtaf = ''.join(nowlist[itemdic[item][3]+1:])
    #置換えデータと結合 直前の\nは取り除く.rstip()
    wtxt = wtxtbf.rstrip() + wtxt + '\n' + wtxtaf
    #設定ファイルを上書き
    try:
      with open(fpath,mode='wt') as fp:
        fp.write(wtxt)
    except Exception as err:
      msb.showerror('error',err)
      return False
  return True

#------------------------------------------------------
if __name__=='__main__':
  root = Tk()

  txt = '新しい項目と値'
  res = update_item('test.conf','item７',txt)
  print(res)
  #dic = read_item_all('test.conf','item4')
  #print(dic)

  root.destroy()
